'use strict';

const _ = require('underscore');

const promises = require('../promises');
const xml = require('../xml');

exports.writeStyleMap = writeStyleMap;
exports.readStyleMap = readStyleMap;


const schema = 'http://schemas.zwobble.org/mammoth/style-map';
const styleMapPath = 'mammoth/style-map';
const styleMapAbsolutePath = '/' + styleMapPath;

function writeStyleMap(docxFile, styleMap) {
  docxFile.write(styleMapPath, styleMap);
  return updateRelationships(docxFile).then(function() {
    return updateContentTypes(docxFile);
  });
}

function updateRelationships(docxFile) {
  const path = 'word/_rels/document.xml.rels';
  const relationshipsUri = 'http://schemas.openxmlformats.org/package/2006/relationships';
  const relationshipElementName = '{' + relationshipsUri + '}Relationship';
  return docxFile.read(path, 'utf8')
    .then(xml.readString)
    .then(function(relationshipsContainer) {
      const relationships = relationshipsContainer.children;
      addOrUpdateElement(relationships, relationshipElementName, 'Id', {
        Id: 'rMammothStyleMap',
        Type: schema,
        Target: styleMapAbsolutePath,
      });

      const namespaces = { '': relationshipsUri };
      return docxFile.write(path, xml.writeString(relationshipsContainer, namespaces));
    });
}

function updateContentTypes(docxFile) {
  const path = '[Content_Types].xml';
  const contentTypesUri = 'http://schemas.openxmlformats.org/package/2006/content-types';
  const overrideName = '{' + contentTypesUri + '}Override';
  return docxFile.read(path, 'utf8')
    .then(xml.readString)
    .then(function(typesElement) {
      const children = typesElement.children;
      addOrUpdateElement(children, overrideName, 'PartName', {
        PartName: styleMapAbsolutePath,
        ContentType: 'text/prs.mammoth.style-map',
      });
      const namespaces = { '': contentTypesUri };
      return docxFile.write(path, xml.writeString(typesElement, namespaces));
    });
}

function addOrUpdateElement(elements, name, identifyingAttribute, attributes) {
  const existingElement = _.find(elements, function(element) {
    return element.name === name &&
            element.attributes[identifyingAttribute] === attributes[identifyingAttribute];
  });
  if (existingElement) {
    existingElement.attributes = attributes;
  } else {
    elements.push(xml.element(name, attributes));
  }
}

function readStyleMap(docxFile) {
  if (docxFile.exists(styleMapPath)) {
    return docxFile.read(styleMapPath, 'utf8');
  }
  return promises.resolve(null);
}
